# Office Bitness Checker Script
# Author: Dario Pascoal
#
# Description: This PowerShell script checks the bitness (32-bit vs 64-bit) of 
# Microsoft Office installations and Access Database Engine components on the 
# current system. This information is critical for database connectivity and 
# automation script compatibility.
#
# The script performs comprehensive checks to determine:
# 1. Operating system architecture (32-bit or 64-bit Windows)
# 2. PowerShell process architecture (important for script execution context)
# 3. Microsoft Office installation bitness (affects automation capabilities)
# 4. Access Database Engine bitness (critical for database connections)
# 5. Available database provider DLLs (determines connection capabilities)
#
# This information helps troubleshoot common issues like:
# - "Provider is not registered on the local machine" errors
# - VBScript/PowerShell database connection failures
# - Office automation compatibility problems
# - Mixed bitness environment conflicts
#
# Business Context: In enterprise environments, it's common to have mixed
# architectures where some components are 32-bit and others are 64-bit.
# This can cause automation scripts to fail when they try to connect to
# databases or interact with Office applications. This script helps identify
# these compatibility issues.
#
# Prerequisites: 
# - Windows PowerShell or PowerShell Core
# - Administrator privileges recommended for complete registry access
# - Microsoft Office and/or Access Database Engine should be installed
#
# Usage: Simply run this script in PowerShell to get a comprehensive report

# =============================================================================
# SECTION 1: SCRIPT INITIALIZATION AND SYSTEM ARCHITECTURE CHECK
# =============================================================================
# This section provides a clear header and checks the fundamental architecture
# of the system. Understanding the OS and PowerShell process bitness is crucial
# because it affects how the script can access registry keys and system files.

Write-Host "=========================================================="
Write-Host "   Microsoft Office and Access Database Engine Checker    "
Write-Host "=========================================================="
Write-Host ""

# Check OS architecture - this tells us if Windows itself is 32-bit or 64-bit
# On 64-bit Windows, we can have both 32-bit and 64-bit applications installed
# On 32-bit Windows, we can only have 32-bit applications
$osArch = if ([Environment]::Is64BitOperatingSystem) { "64-bit" } else { "32-bit" }
Write-Host "Operating System: $osArch Windows"

# Check PowerShell process architecture - this is important because:
# - 64-bit PowerShell can access both 64-bit and 32-bit registry hives
# - 32-bit PowerShell can only access 32-bit registry hives directly
# - This affects which installed components we can detect
$psArch = if ([Environment]::Is64BitProcess) { "64-bit" } else { "32-bit" }
Write-Host "PowerShell Process: $psArch"
Write-Host ""

# =============================================================================
# SECTION 2: MICROSOFT OFFICE INSTALLATION DETECTION
# =============================================================================
# This section searches the Windows Registry to find Microsoft Office installations.
# We check both 64-bit and 32-bit registry locations because Office can be
# installed in either architecture regardless of the OS bitness.

# Check Microsoft Office installation
Write-Host "Checking Microsoft Office installations..."

# Define registry locations where Office information is stored
# The main location contains 64-bit installations
# The Wow6432Node location contains 32-bit installations on 64-bit Windows
$officeLocations = @(
    "HKLM:\SOFTWARE\Microsoft\Office",                    # 64-bit Office installations
    "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office"         # 32-bit Office installations
)

$officeFound = $false

# Loop through each registry location to find Office installations
foreach ($location in $officeLocations) {
    if (Test-Path $location) {
        # Determine the bitness based on the registry path
        # Wow6432Node is where Windows stores 32-bit application info on 64-bit systems
        $officeArch = if ($location -like "*Wow6432Node*") { "32-bit" } else { "64-bit" }
        
        # Check for version specific paths - Office versions have specific version numbers
        # Each major Office release has a unique version number that helps identify it
        $versions = @("16.0", "15.0", "14.0", "12.0", "11.0")
        
        # Loop through each version number to see what's installed
        foreach ($version in $versions) {
            $versionPath = Join-Path $location $version
            if (Test-Path $versionPath) {
                $officeFound = $true
                
                # Create a mapping of version numbers to user-friendly names
                # This helps users understand which Office version they have
                $versionMap = @{
                    "16.0" = "Office 2016/2019/2021/365"   # Modern Office versions
                    "15.0" = "Office 2013"                 # Office 2013
                    "14.0" = "Office 2010"                 # Office 2010
                    "12.0" = "Office 2007"                 # Office 2007
                    "11.0" = "Office 2003"                 # Legacy Office 2003
                }
                $versionLabel = $versionMap[$version]
                Write-Host "  - Found $officeArch $versionLabel (Version $version)"
            }
        }
    }
}

# If we didn't find any Office installations, let the user know
if (-not $officeFound) {
    Write-Host "  - No Microsoft Office installation detected in registry"
}

Write-Host ""

# =============================================================================
# SECTION 3: ACCESS DATABASE ENGINE DETECTION (REGISTRY METHOD)
# =============================================================================
# The Access Database Engine (ACE) is crucial for connecting to Access databases
# and Excel files from automation scripts. This section checks for ACE installations
# by looking in specific registry locations where ACE registers itself.

# Check Access Database Engine installation
Write-Host "Checking Microsoft Access Database Engine installation..."

# Define specific registry paths where Access Connectivity Engine is registered
# We check multiple versions and both 32-bit and 64-bit locations
# ACE is what allows applications to read/write Access databases and Excel files
$aceLocations = @(
    # 64-bit ACE installations (stored in main registry hive)
    @{ Path = "HKLM:\SOFTWARE\Microsoft\Office\16.0\Access Connectivity Engine"; Bitness = "64-bit"; Name = "ACE 16.0" },
    @{ Path = "HKLM:\SOFTWARE\Microsoft\Office\15.0\Access Connectivity Engine"; Bitness = "64-bit"; Name = "ACE 15.0" },
    @{ Path = "HKLM:\SOFTWARE\Microsoft\Office\14.0\Access Connectivity Engine"; Bitness = "64-bit"; Name = "ACE 14.0" },
    @{ Path = "HKLM:\SOFTWARE\Microsoft\Office\12.0\Access Connectivity Engine"; Bitness = "64-bit"; Name = "ACE 12.0" },
    # 32-bit ACE installations (stored in Wow6432Node on 64-bit systems)
    @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office\16.0\Access Connectivity Engine"; Bitness = "32-bit"; Name = "ACE 16.0" },
    @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office\15.0\Access Connectivity Engine"; Bitness = "32-bit"; Name = "ACE 15.0" },
    @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office\14.0\Access Connectivity Engine"; Bitness = "32-bit"; Name = "ACE 14.0" },
    @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office\12.0\Access Connectivity Engine"; Bitness = "32-bit"; Name = "ACE 12.0" }
)

$aceFound = $false

# Check each potential ACE installation location
foreach ($ace in $aceLocations) {
    if (Test-Path $ace.Path) {
        $aceFound = $true
        Write-Host "  - Found $($ace.Bitness) $($ace.Name)"
    }
}

# =============================================================================
# SECTION 4: ACCESS DATABASE ENGINE DETECTION (UNINSTALL REGISTRY METHOD)
# =============================================================================
# Sometimes ACE installations don't appear in the Office registry locations,
# especially for standalone installations. The Windows uninstall registry
# provides an alternative way to detect installed software.

# Also check via the uninstall keys which works better for standalone installs
Write-Host ""
Write-Host "Checking via Uninstall registry keys..."

# Define the Windows uninstall registry locations
# These locations contain information about all installed programs
# including standalone Access Database Engine installations
$uninstallLocations = @(
    @{ Path = "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall"; Bitness = "64-bit" },           # 64-bit software
    @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall"; Bitness = "32-bit" } # 32-bit software
)

$aceFoundInUninstall = $false

# Search through uninstall entries for Access Database Engine
foreach ($location in $uninstallLocations) {
    # Get all uninstall entries and filter for Access Database Engine
    # This command looks through all installed program entries and finds ones
    # that have "Access Database Engine" in their display name
    $keys = Get-ItemProperty ($location.Path + "\*") -ErrorAction SilentlyContinue | 
            Where-Object { $_.DisplayName -like "*Access Database Engine*" } |
            Select-Object DisplayName, DisplayVersion

    # Display any found Access Database Engine installations
    foreach ($key in $keys) {
        $aceFoundInUninstall = $true
        Write-Host "  - Found $($location.Bitness) $($key.DisplayName) (Version $($key.DisplayVersion))"
    }
}

# =============================================================================
# SECTION 5: MICROSOFT ACCESS EXECUTABLE DETECTION
# =============================================================================
# This section looks for the actual Microsoft Access executable file (MSACCESS.EXE)
# in common installation locations. Finding the executable and determining its
# bitness provides definitive proof of what version of Access is installed.

# Final check for the MSAccess.exe executable if Office is present
# We check multiple common installation paths because Office can be installed
# in different locations depending on the installation method and version
$possiblePaths = @(
    # Office 365/2019/2021 Click-to-Run installations
    "${env:ProgramFiles}\Microsoft Office\root\Office16\MSACCESS.EXE",           # 64-bit C2R
    "${env:ProgramFiles(x86)}\Microsoft Office\root\Office16\MSACCESS.EXE",      # 32-bit C2R
    # Traditional MSI installations
    "${env:ProgramFiles}\Microsoft Office\Office16\MSACCESS.EXE",               # 64-bit MSI Office 2016+
    "${env:ProgramFiles(x86)}\Microsoft Office\Office16\MSACCESS.EXE",          # 32-bit MSI Office 2016+
    # Office 2013 installations
    "${env:ProgramFiles}\Microsoft Office\root\Office15\MSACCESS.EXE",          # 64-bit C2R Office 2013
    "${env:ProgramFiles(x86)}\Microsoft Office\root\Office15\MSACCESS.EXE",     # 32-bit C2R Office 2013
    "${env:ProgramFiles}\Microsoft Office\Office15\MSACCESS.EXE",               # 64-bit MSI Office 2013
    "${env:ProgramFiles(x86)}\Microsoft Office\Office15\MSACCESS.EXE",          # 32-bit MSI Office 2013
    # Office 2010 installations
    "${env:ProgramFiles}\Microsoft Office\Office14\MSACCESS.EXE",               # 64-bit Office 2010
    "${env:ProgramFiles(x86)}\Microsoft Office\Office14\MSACCESS.EXE"           # 32-bit Office 2010
)

Write-Host ""
Write-Host "Checking for Access executable..."
foreach ($path in $possiblePaths) {
    if (Test-Path $path) {
        $fileInfo = Get-Item $path
        
        # Determine the bitness of the executable by examining its PE header
        # The PE (Portable Executable) header contains machine type information
        # that tells us if the file is compiled for 32-bit or 64-bit architecture
        $bitness = if ((Get-Command Get-ProcessArchitecture -ErrorAction SilentlyContinue)) {
            # Use built-in command if available (PowerShell 7+)
            (Get-ProcessArchitecture $path)
        } else {
            # Check PE header manually to determine bitness
            # This reads the binary file structure to determine the target architecture
            $stream = New-Object System.IO.FileStream($path, [System.IO.FileMode]::Open, [System.IO.FileAccess]::Read)
            try {
                $reader = New-Object System.IO.BinaryReader($stream)
                # Navigate to the PE header offset (located at position 0x3C in the file)
                $stream.Position = 0x3C
                $peOffset = $reader.ReadInt32()
                # Read the machine type from the PE header
                $stream.Position = $peOffset + 4
                $machineType = $reader.ReadInt16()
                
                # Interpret the machine type value
                # 0x014c = IMAGE_FILE_MACHINE_I386 (32-bit Intel)
                # 0x8664 = IMAGE_FILE_MACHINE_AMD64 (64-bit x64)
                switch ($machineType) {
                    0x014c { "32-bit" }
                    0x8664 { "64-bit" }
                    default { "Unknown" }
                }
            } finally {
                # Always close file handles to prevent locks
                $reader.Close()
                $stream.Close()
            }
        }
        
        Write-Host "  - Found MS Access executable: $path ($bitness)"
    }
}

# Inform user if no Access Database Engine was found anywhere
if (-not ($aceFound -or $aceFoundInUninstall)) {
    Write-Host "  - No Microsoft Access Database Engine detected in registry"
}

Write-Host ""

# =============================================================================
# SECTION 6: DATABASE PROVIDER DLL VERIFICATION
# =============================================================================
# This section checks for the actual DLL files that provide database connectivity.
# These are the files that applications use to connect to Access databases and
# Excel files. Their presence and bitness determines what connections are possible.

Write-Host "Provider DLL Information:"

# Define the critical database provider DLLs and their locations
# These DLLs are what actually handle the database connections
$providerPaths = @(
    # Legacy Microsoft Jet providers (older Access databases)
    @{ Path = "${env:SystemRoot}\System32\msjetoledb40.dll"; Name = "Microsoft Jet OLEDB 4.0" },
    @{ Path = "${env:SystemRoot}\System32\msrd3x40.dll"; Name = "Microsoft Jet Red 3.x" },
    @{ Path = "${env:SystemRoot}\SysWOW64\msjetoledb40.dll"; Name = "Microsoft Jet OLEDB 4.0 (32-bit)" },
    @{ Path = "${env:SystemRoot}\SysWOW64\msrd3x40.dll"; Name = "Microsoft Jet Red 3.x (32-bit)" },
    # Modern ACE providers (newer Access databases and Excel files)
    @{ Path = "${env:SystemRoot}\System32\aceoledb.dll"; Name = "Microsoft ACE OLEDB" },
    @{ Path = "${env:SystemRoot}\SysWOW64\aceoledb.dll"; Name = "Microsoft ACE OLEDB (32-bit)" }
)

# Check each provider DLL to see if it exists and what version it is
foreach ($provider in $providerPaths) {
    if (Test-Path $provider.Path) {
        # Get the file version information for troubleshooting purposes
        $fileVersion = (Get-Item $provider.Path).VersionInfo.FileVersion
        Write-Host "  - $($provider.Name): Found - Version $fileVersion"
    } else {
        Write-Host "  - $($provider.Name): Not found"
    }
}

Write-Host ""

# =============================================================================
# SECTION 7: RECOMMENDATIONS AND TROUBLESHOOTING GUIDANCE
# =============================================================================
# This final section provides actionable recommendations based on the findings.
# It helps users understand how to resolve common compatibility issues and
# configure their environment for successful database automation.

Write-Host "=========================================================="
Write-Host "                     Recommendations                      " 
Write-Host "=========================================================="
Write-Host ""

# Provide guidance for 64-bit Windows environments
Write-Host "If you're running a 64-bit version of Windows:"
Write-Host "1. Ensure the bitness of Microsoft Access Database Engine matches the bitness of your Office installation"
Write-Host "2. If using a 64-bit process to access data, you need 64-bit providers"
Write-Host "3. If using a 32-bit process to access data, you need 32-bit providers"
Write-Host ""

# Provide specific guidance for VBScript automation issues
Write-Host "For your VBScript issue:"
Write-Host "1. If running in 64-bit mode, either:"
Write-Host "   - Install the 64-bit Access Database Engine (if it doesn't conflict with Office bitness)"
Write-Host "   - Force the script to run in 32-bit mode using 'cscript /nologo /32'"
Write-Host "2. If running in 32-bit mode and still having issues, check permissions to the Access database"
Write-Host ""

# Important compatibility warning
Write-Host "NOTE: It's generally recommended that Office and Access Database Engine have the same bitness."
Write-Host "      Installing mismatched bitness may cause conflicts or require special installation parameters."
